﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/omics/OmicsRequest.h>
#include <aws/omics/Omics_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Omics {
namespace Model {

/**
 */
class CreateRunGroupRequest : public OmicsRequest {
 public:
  AWS_OMICS_API CreateRunGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRunGroup"; }

  AWS_OMICS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A name for the group.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateRunGroupRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of CPUs that can run concurrently across all active runs
   * in the run group.</p>
   */
  inline int GetMaxCpus() const { return m_maxCpus; }
  inline bool MaxCpusHasBeenSet() const { return m_maxCpusHasBeenSet; }
  inline void SetMaxCpus(int value) {
    m_maxCpusHasBeenSet = true;
    m_maxCpus = value;
  }
  inline CreateRunGroupRequest& WithMaxCpus(int value) {
    SetMaxCpus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of runs that can be running at the same time.</p>
   */
  inline int GetMaxRuns() const { return m_maxRuns; }
  inline bool MaxRunsHasBeenSet() const { return m_maxRunsHasBeenSet; }
  inline void SetMaxRuns(int value) {
    m_maxRunsHasBeenSet = true;
    m_maxRuns = value;
  }
  inline CreateRunGroupRequest& WithMaxRuns(int value) {
    SetMaxRuns(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum time for each run (in minutes). If a run exceeds the maximum run
   * time, the run fails automatically.</p>
   */
  inline int GetMaxDuration() const { return m_maxDuration; }
  inline bool MaxDurationHasBeenSet() const { return m_maxDurationHasBeenSet; }
  inline void SetMaxDuration(int value) {
    m_maxDurationHasBeenSet = true;
    m_maxDuration = value;
  }
  inline CreateRunGroupRequest& WithMaxDuration(int value) {
    SetMaxDuration(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags for the group.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateRunGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateRunGroupRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>To ensure that requests don't run multiple times, specify a unique ID for
   * each request.</p>
   */
  inline const Aws::String& GetRequestId() const { return m_requestId; }
  inline bool RequestIdHasBeenSet() const { return m_requestIdHasBeenSet; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  CreateRunGroupRequest& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of GPUs that can run concurrently across all active runs
   * in the run group.</p>
   */
  inline int GetMaxGpus() const { return m_maxGpus; }
  inline bool MaxGpusHasBeenSet() const { return m_maxGpusHasBeenSet; }
  inline void SetMaxGpus(int value) {
    m_maxGpusHasBeenSet = true;
    m_maxGpus = value;
  }
  inline CreateRunGroupRequest& WithMaxGpus(int value) {
    SetMaxGpus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  int m_maxCpus{0};
  bool m_maxCpusHasBeenSet = false;

  int m_maxRuns{0};
  bool m_maxRunsHasBeenSet = false;

  int m_maxDuration{0};
  bool m_maxDurationHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_requestId{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_requestIdHasBeenSet = true;

  int m_maxGpus{0};
  bool m_maxGpusHasBeenSet = false;
};

}  // namespace Model
}  // namespace Omics
}  // namespace Aws
